;
;	REDIR - Character I/O Redirection via IOBYTE
;		Entry, DI = address of device redirection table.
;			CL = count to shift IOBYTE.
;
;	IOBYTE:  | LST | PUN | RDR | CON | , 2 bits each.
;
;	Note:  To redirect console in to serial port, IOBYTE must
;		be either 01 for COM1 or 10 for COM2.
;
REDIR:	EQU	$
	MOV	SI,IOBYTE	;point to IOBYTE
	PUSH	DS		;save DS
	MOV	DS,RPMSEG	;get RPM segment in DS
	MOV	DL,[SI]		;get IOBYTE
	POP	DS		;restore DS
	ROR	DL,CL		;shift by count received
	AND	DL,03H		;mask result
;
	MOV	DH,0
	ADD	DI,DX		;add result to table address
	MOV	AH,0		;initialize AH
	MOV	AL,[DI]		;get redirection ordinal
	AND	AL,0FH		;mask to table ordinal limit.
	SHL	AX		;multiply by two
	MOV	BX,OFFSET PRCTBL	;address of processor table
	ADD	BX,AX		;add to jump table address
	MOV	DADRS,BX	;save result
	JMP	[BX]		;dispatch directed processor
;
;	I/O Redirection Processors Table
;
PRCTBL:	EQU	$
;
	.WORD	CRTOUT1		; 0 = Console Display (CRT) char out.
	.WORD	KEYIN1		; 1 = Console Keyboard character in.
	.WORD	LPT1OT		; 2 = Line Printer 1 character out.
	.WORD 	LPT2OT		; 3 = Line Printer 2 character out.
	.WORD	LPT3OT		; 4 = Line Printer 3 character out.
	.WORD	COM1IN		; 5 = Comm Port 1 character in.
	.WORD	COM1OT		; 6 = Comm Port 1 character out.
	.WORD	COM2IN		; 7 = Comm Port 2 character in.
	.WORD	COM2OT		; 8 = Comm Port 2 character out.
	.WORD	COM1SU		; 9 = Comm Port 1 Setup
	.WORD	COM2SU		; A = Comm Port 2 Setup
	.WORD	COM1TS		; B = Comm Port 1 Test (Status)
	.WORD	COM2TS		; C = Comm Port 2 Test (Status)
	.WORD	EXIT		; D = spare.
	.WORD	EXIT		; E = spare.
	.WORD	EXIT		; F = spare.
;
DADRS:	.WORD	0		;current processor dispatched.
;
;	RS-232 Serial Communications I/O ( COM1 and COM2 ports).
;
COM1SU:	EQU	$	;Comm Port 1 Setup (Initialization).
	MOV	DX,0	;address Comm Port 1
	JMP	COMSET
COM2SU:	EQU	$	;Comm Port 2 Setup (Initialization).
	MOV	DX,1	;address Comm Port 2
COMSET:	EQU	$
	MOV	AL,CIN	;get setup byte from caller
	MOV	AH,0	;Comm Port initialization operation
	JMP	COMM	;go do it
;
COM1TS:	EQU	$	;Comm Port 1 Test Status (Line, Modem)
	MOV	DX,0	;Address Comm Port 1
	JMP	COMTST
COM2TS:	EQU	$	;Comm Port 2 Test Status (Line, Modem)
	MOV	DX,1	;Address Comm Port 2
COMTST:	EQU	$
	MOV	AH,3	;Comm Port Test Line/Modem Status operation.
	JMP	COMM
;
COM1IN:	EQU	$	;Comm Port 1 Character in (Receive)
	MOV	DX,0	;address Comm Port 1
	JMP	COMIN
COM2IN:	EQU	$	;Comm Port 2 Character in (Receive)
	MOV	DX,1	;address Comm Port 2
COMIN:	EQU	$
	MOV	AH,2	;receive character operation
	INT	014H	;ROM BIOS interrupt.
	MOV	COMSTS,AX	;save character and status
	JMP	EXIT	;return character to caller.
;
COM1OT:	EQU	$	;Comm Port 1 Character out (Transmit)
	MOV	DX,0	;address Comm Port 1
	JMP	COMOUT
COM2OT:	EQU	$	;Comm Port 2 Character out (Transmit)
	MOV	DX,1	;address Comm Port 2
COMOUT:	EQU	$
	MOV	AL,CIN	;pick up character to transmit
	MOV	AH,1	;transmit character operation
COMM:	EQU	$	;communications port I/O
	INT	014H	;ROM BIOS interrupt
	MOV	COMSTS,AX	;save complete status
	MOV	AL,AH	;status to AL (A regis) for caller.
	JMP	EXIT	;we are done
;
;	PUNCH OUT and READER IN Functions.
;
PUNOUT:	EQU	$	;Character out to punch device.
	MOV	CL,4	;IOBYTE shift count
	MOV	DI,OFFSET PUNO	;redirection table address.
	JMP	REDIR	
;
RDRIN:	EQU	$	;Character in from reader device.
	MOV	CL,2	;IOBYTE shift count.
	MOV	DI,OFFSET RDRI	;redirection table address.
	JMP	REDIR
;
SETLN1:		CALL	MULT10		;multiply/store first nibble
		MOV	LINPOS,DL
		JMP	MATCH
SETLN2:		AND	CL,0FH		;mask & add second nibble
		ADD	LINPOS,CL
		JMP	SETLIN1		;finish set line position
;
SETCL1:		CALL	MULT10		;multiply/store first nibble
		MOV	COLPOS,DL
		JMP	MATCH
SETCL2:		AND	CL,0FH		;mask & add second nibble
		ADD	COLPOS,CL
		JMP	SETCOL1		;finish set column position
;
;	CRTOUT1 - Character in CL register to
;			output to the display.
;
CRTOUT1:	EQU	$	;character to CRT display.
		MOV	AL,TERMID	;emulation terminal I.D.
		AND	AL,AL		;check if emulation mode
		JZ	CHROUT		;not emulating any terminal
;
		MOV	BYTE PTR DMPSW,0	;clear dump char switch
NXTSTR:		MOV	CL,CIN		;character to display
		MOV	BX,NXTBL	;pointer to next string table.
		MOV	BX,[BX]		;get string table processor address.
      		AND	BX,BX
		JZ      CRTOUT2		;end of strings
		JMP	STRPRC		;string to process
CRTOUT2:	EQU	$		;exit matched string processor
		MOV	BX,TRMTBL	;resets to start of table
		MOV	NXTBL,BX	;to process next character.
		TEST	BYTE PTR DMPSW,01H	;do we dump char?
		JZ	CHROUT		;normal character to display
		JMP	CRTOUTX		;emulation char to dump
CHROUT:		EQU	$		;go display character
		MOV	AL,CIN		;get the character to display
		CALL	TTYOUT		;our own version of ROM BIOS TTY
		JMP	CRTOUTX		;exit.
;
SETFA1:		MOV	FB,CL		;save FA nibble in FB temporarily
		JMP	MATCH
SETFB2:		MOV	CH,[SI]+2	;pick up delimiter character
		CMP	CH,CL		;is this the delimiter?
		JNZ     SETBF3		;no	
  		CALL	INCPTR
  		CALL	INCPTR
  		JMP	PRCLN
SETBF3:		MOV	CH,FB		;get first FA nibble
		MOV	FA,CH		;save it in FA
		MOV	FB,CL		;save FB
		JMP	MATCH
SETFC1:		MOV	FD,CL		;save FC nibble in FD temporarily
		JMP	MATCH
SETFD2:		MOV	CH,[SI]+2	;pick up delimiter character
		CMP	CH,CL		;is this the delimiter?
		JNZ	SETFD3		;no
		CALL	INCPTR
		CALL	INCPTR
		JMP	PRCCL
SETFD3:		MOV	CH,FD		;get first FC nibble
		MOV	FC,CH		;save it in FC
		MOV	FD,CL		;save FD
		JMP	MATCH
;
CHKLN:		CALL	INCPTR
		MOV	DL,[SI]
		CMP	DL,CL		;check if line delimiter
		JZ	PRCLN		;yes, process line position
		JMP	NOMTCH		;no match
;
CHKCL:		CALL	INCPTR
		MOV	DL,[SI]
		CMP	DL,CL		;check if column delimiter
		JZ	PRCCL		;yes, process column position
		JMP	NOMTCH		;no match
;
PRCLN:		MOV	CL,FA		;first line position nibble
		CALL	MULT10
		MOV	LINPOS,DL
		MOV	CL,FB		;second line position nibble
		JMP	SETLN2
;
PRCCL:		MOV	CL,FC		;first column position nibble
		CALL	MULT10
		MOV	COLPOS,DL
		MOV	CL,FD		;second column position nibble
		JMP	SETCL2
;
SETFA1R:	JMP	SETFA1
SETFB2R:	JMP	SETFB2
SETFC1R		JMP	SETFC1
SETFD2R		JMP	SETFD2
;
STRPRC:		EQU	$
		MOV	AX,[BX]		;get string processor address
		AND	AX,AX		;if zero,
		JZ	NXT1		;skip to next string
		MOV	SI,BX		;set up index to string
		ADD	SI,2		;point to char index
		MOV	DI,SI		;save index ptr in DI
		MOV	BH,0
		MOV	BL,[SI]		;pointer to next string character.
		ADD	SI,BX 		;point to next char to process
		MOV	DL,[SI]		;pick up character
		CMP	DL,0FAH		;first nibble, line position
		JZ 	SETFA1R
		CMP	DL,0FBH		;second nibble, line position
		JZ	SETFB2R
		CMP	DL,0FCH		;first nibble, column position
		JZ	SETFC1R
		CMP	DL,0FDH		;second nibble, column position
		JZ	SETFD2R
		CMP	DL,0F8H		;line delimiter?
		JZ	CHKLN		;yes
		CMP	DL,0F9H		;column delimiter?
		JZ	CHKCL		;yes
		CMP	DL,0FEH		;is this line position?
		JZ	SETLIN 		;yes
       		CMP	DL,0FFH		;is this column position?
		JZ 	SETCOL 		;yes
		CMP	DL,CL		;check for match
		JZ	MATCH		;yes
NOMTCH:		MOV	BYTE PTR [DI],1	;no match, reset string index
NXT1:		ADD	NXTBL,2 	;bump table to next string
		JMP	NXTSTR		;process next string
;
;	CURSOR - Set Cursor Position Processor
;
CURSOR:		EQU	$
		MOV	AH,0FH		;get current page in BH
		INT	010H
		MOV	DX,COLPOS	;get new cursor line,column values
CURSOR1:	EQU	$
		MOV	AH,02H		;cursor position request to
		INT	010H		;ROM BIOS video.
		RET
;
FA:		.BYTE	0		;first line position nibble
FB:		.BYTE	0		;second line position nibble
FC:		.BYTE	0		;first column position nibble
FD:		.BYTE	0		;second line position nibble
COLPOS:		.BYTE	0		;DL = cursor column position.
LINPOS:		.BYTE	0		;DH = cursor line position.
NXTBL:		.WORD	STRTBL		;address of next table entry.
DMPSW:		.BYTE	0		;dump character switch
;
;	CLEAR - Clear Screen and Home Cursor Processor
;
CLEAR:	EQU	$
		MOV	AH,0FH		;get current page number
		INT	010H	
		MOV	AH,8		;get char attribute
		INT	010H
		MOV	BH,AH		;into BH
		MOV	AX,0600H	;scroll up & blank screen
		MOV	CX,0
		MOV	DX,01950H	;25 lines, 80 columns
		INT	010H
HOME:	EQU	$
		MOV	AH,0FH		;get current page in BH
		INT	010H
		MOV	DX,0		;cursor to home
		MOV	AH,02H
		INT	010H
NULSTR:		RET			;done / null string exit
;
SETLIN:		MOV	LINPOS,CL
SETLIN1:	CALL	INCPTR		;address bias byte
		MOV	DL,[SI]		;get bias byte
		SUB	LINPOS,DL	;remove bias
		JMP	MATCH
;
SETCOL:		MOV	COLPOS,CL	;column position value
SETCOL1:	CALL	INCPTR		;address bias byte
		MOV	DL,[SI]		;get bias byte
		SUB	COLPOS,DL	;remove bias
;
MATCH:		EQU	$
		MOV	DMPSW,01H	;set the dump char switch.
		CALL	INCPTR 		;bump string index
		MOV	DL,[SI]		;get next char in string
		CMP	DL,0
		JZ	CALLPRC		;total match, call process
		JMP	NXT1		;not done, bump to next string
;
;
INCPTR:		INC	BYTE PTR [DI]	;bump string index
		INC	SI		;bump our pointer
		RET
;
CALLPRC:	EQU	$		;call string processor.
		MOV	BYTE PTR [DI],1	;reset string index
		CALL	AX		;process the string function.
		MOV	FA,0		;clear cursor positioning nibbles
		MOV	FB,0
		MOV	FC,0
		MOV	FD,0
		MOV	BX,TRMTBL	;reset to start of table
		MOV	NXTBL,BX	;to process next character
RSINDX:		MOV	SI,[BX]		;reset all string pointers
		AND	SI,SI
		JNZ	RSINDX1		;another string
		JMP	CRTOUT2		;done
RSINDX1:	ADD	SI,2		;index address
		MOV	BYTE PTR [SI],1	;reset index
		ADD	BX,2		;next string	
 		JMP	RSINDX 		;loop until all index ptrs reset
;
MULT10:		AND	CL,0FH		;mask ASCII byte
		MOV	DL,CL		;move byte to DL
		ROL	DL
		ROL	DL		;*4
		ADD	DL,CL		;*5 (add saved byte)
		ROL	DL		;*10
		RET
;
GETCUR:		MOV	AH,0FH		;Get current page in BH.
		INT	010H
       		MOV	AH,03H		;Get cursor coordinates in DX
		INT	010H
		RET
;
;	INSERT LINE at current cursor position.
;
INSERT:		CALL	GETCUR		;Cursor coordinates in DX.
		MOV	CH,DH		;Line,
		MOV	CL,0		;Column, Upper Left corner		
		MOV	DH,017H		;Line,
		MOV	DL,04FH		;Column, Lower Right corner
		MOV	AL,01H		;One blank line at bottom of screen.
		MOV	BH,ATTRIB	;Character attribute byte.
		MOV	AH,07H		;Scroll page down.
		INT	010H
		RET
;
;	DELETE LINE at current cursor position.
;
DELETE:		CALL	GETCUR		;Cursor coordinates in DX.
		MOV	CH,DH		;Line,
		MOV	CL,0		;Column, upper left corner.
		MOV	DH,017H		;Line,
		MOV	DL,04FH		;Column, lower right corner.
		MOV	AL,01H		;One blank line at bottom of screen.
		MOV	BH,ATTRIB	;Character attribute byte.
		MOV	AH,06H		;Scroll page up.
		INT	010H
		RET
;
;	Set Bright Intensity.
;
INTBRITE:	MOV	AL,ATTRIB	;Get current attribute byte.
		OR	AL,08H		;Set bright intensity bit.
		MOV	BYTE PTR ATTRIB,AL
		RET
;
;	Set Normal Intensity.
;
INTNORM:	MOV	AL,ATTRIB	;Get current attribute byte.
		AND	AL,0F7H		;Mask off bright intensity bit.
		MOV	BYTE PTR ATTRIB,AL
		RET
;
;	Toggle the video (reverses background/foreground colors).
;
VIDREV:  	MOV	AL,ATTRIB	;Get current attribute byte.
		MOV	PREVATT,AL	;Save previous attribute byte.
		MOV	AH,AL
		AND	AH,088H		;Preserve blinking/bright bits.
		MOV	CL,4
		ROR	AL,CL  		;Swap attribute nibbles.
		AND	AL,077H		;Mask off blinking/bright bits.
		OR	AL,AH		;Set in saved blinking/bright bits.
		MOV	BYTE PTR ATTRIB,AL
		RET
;
;	Set Normal (Initial) Video Attribute Status.          
;
NORMAL:		MOV	AL,NRMATTR	;Get previous attribute byte.
		MOV	ATTRIB,AL	;Restore to current attribute.
		RET
;
;	Set Black & White Video (Monochrome).
;
BWVIDEO:	MOV	AL,07H   	;Set Black & White Video bits.
		MOV	BYTE PTR ATTRIB,AL
		RET
;
;	Set Blinking text on.
;
BLINKON:	MOV	AL,ATTRIB	;Get current attribute byte.
		OR	AL,080H		;Set Blinking bit.
		MOV	BYTE PTR ATTRIB,AL
		RET		
;
;	Shut off Blinking text.
;
BLINKOFF:	MOV	AL,ATTRIB	;Get current attribute byte.
		AND	AL,07FH		;Turn Blinking bit off.
		MOV	BYTE PTR ATTRIB,AL
		RET
;
;	Set Underlined character on (monochrome board only).
;
UNDLNON:	MOV	AL,ATTRIB	;Get current attribute byte.
		MOV	PREVATT,AL	;Save it for removing underline.
		AND	AL,088H		;Preserve blinking/bright bits.
		OR	AL,01H		;Set underline character bit.
		MOV	BYTE PTR ATTRIB,AL
		RET
;
;	Shut underlined character off (monochrome board only).
;
UNDLNOFF:	MOV	AL,ATTRIB	;Get current attribute byte.
		MOV	AH,PREVATT	;Get attribute before underline. 
		AND	AH,077H
		AND	AL,088H		;Preserve blinking/bright bits.
		OR 	AL,AH
		MOV	BYTE PTR ATTRIB,AL
		RET
;
;	Remember current cursor position.
;
REMCUR:		CALL	GETCUR		;Get cursor position in DX.
		MOV	WORD PTR CURSAV,DX
		RET
;
;	Restore previous cursor position remembered.
;
RESTCUR:	MOV	DX,WORD PTR CURSAV	;Get remembered position.
		MOV	WORD PTR COLPOS,DX
		CALL	CURSOR
		RET		
;
;	Set Block type cursor.
;
BLKCUR:		CALL	GETCUR			;Get status/cursor info.
		MOV	CX,020AH		;Monochrome block cursor
		CMP	AL,07H			;see if monochrome board
		JZ	SETCURS
		MOV	CX,0106H		;Color card block cursor
		CMP	AL,02H			;B&W 80X25 text mode.
		JZ	SETCURS
		CMP	AL,03H			;Color 80X25 text mode.
		JZ	SETCURS
		RET
;
;	Set Line type cursor.
;
LINCUR:		CALL	GETCUR			;Get status/cursor info.
		MOV	CX,0B0CH		:Monochrome underline cursor.
		CMP	AL,07H			;see if monochrome card
		JZ	SETCURS
		MOV	CX,0607H		;Color card underline cursor.
		CMP	AL,02H			;B&W 80X25 text mode
		JZ	SETCURS
		CMP	AL,03H			;Color 80X25 text mode.
		JZ	SETCURS
		RET
SETCURS:	MOV	AH,01H			;Set cursor type.
		RET		
;
;	New Line (CR/LF)
;
NEWLINE:	CALL	GETCUR			;Get cursor position.
		XOR	DL,DL			;zero column position.
		CALL	CURSOR1			;do it
		MOV	BYTE PTR DMPSW,0	;clear dump char switch
		MOV	CL,0AH
		MOV	CIN,CL			;output line-feed
		RET
;
;	Erase to End-of-line.
;
ERASEOL:	CALL	GETCUR			;Get cursor position.
		MOV	CL,050H			;80 columns
		SUB	CL,DL			;less where we are
		XOR	CH,CH
		OR	CX,CX			;see how many to blank
		JZ	EOLX			;none
		MOV	BL,ATTRIB		;current attribute
		MOV	AL,020H			;blank space
		MOV	AH,09H			;TTY output function.
		INT	010H			
EOLX:		RET
;
;	Erase to End-of-screen.
;
ERASEOS:	CALL	ERASEOL			;Erase to end of current line.
		CALL	GETCUR			;Get cursor position.
		CMP	DH,BYTE PTR SCRLIN	;bottom line?
		JZ	EOSX			;Yes, done.
		INC	DH			;Increment line number.
		MOV	CX,DX
		MOV	CL,0			;upper left corner
		MOV	DH,SCRLIN		;bottom line
		MOV	DL,04FH			;column 79 (lower rt corner)
		MOV	AL,0
		MOV	BH,ATTRIB		;attribute byte
		MOV	AH,06H			;scroll page up
		INT	010H
EOSX:		RET	
;
;	Cursor Right one column.
;
RIGHTCUR:	CALL	GETCUR			;Get cursor position.
		CMP	DL,04FH			;At 79th column?
		JZ	EOSX			;done
		INC	DL			;increment column
		CALL	CURSOR1			;do it
		RET
;
;	Cursor Down one line.
;
DOWNCUR:	CALL	GETCUR			;Get cursor position.
		CMP	DH,BYTE PTR SCRLIN	;At bottom line?
		JZ	EOSX			;done
		INC	DH			;increment line
		CALL	CURSOR1			;do it
		RET		
;
;	Cursor Up one line.
;
UPCUR:		CALL	GETCUR			;Get cursor position.
		CMP	DH,0			;At top of screen?
		JZ	EOSX			;done
		DEC	DH			;decrement line number
		CALL	CURSOR1			;do it
		RET
;
;	Teletype Output Subroutine (CBIOS88 version of ROM BIOS TTY OUT)
;
TTYOUT:		PUSH	AX			;save character to display
		CALL	GETCUR			;get cursor position in DX
		POP	AX			;recover character
		CMP	AL,08H			;backspace
		JZ	BSPACE
		CMP	AL,0DH			;carriage-return
		JZ	CRET
		CMP	AL,0AH			;line-feed
		JZ	LFEED
		CMP	AL,07H			;bell
		JZ	BELL
;
OUTACHR:	MOV	BL,ATTRIB		;current attribute
		MOV	CX,1			;output one character from AL
		MOV	AH,09H			;write attrib and character
		AND	AL,OUTMSK		;mask character
		INT	010H
		INC	DL			;increment column
		CMP	DL,07FH			;test for end of line
		JNZ	SETCRS			;no - set cursor
		MOV	DL,0			;reset column if EOL
		CMP	DH,BYTE PTR SCRLIN	;check if bottom line
		JNB	SCROLL			;yes 
INCLIN:		INC	DH			;increment line
SETCRS:		CALL	CURSOR1			;set cursor for next character
		RET		
;
SCROLL:		CALL	CURSOR1			;set cursor
SCROLL1:	MOV	BH,ATTRIB		;current attribute
		MOV	CX,0			;upper left corner
		MOV	DH,SCRLIN		;bottom line
		MOV	DL,04FH			;column 79
		MOV	AL,01H			;scroll one line
		MOV	AH,06H			;scroll up
		INT	010H
		RET
;
BSPACE:		CMP	DL,0			;check end of line
		JZ	SETCRS			;yes
		DEC	DL			;decrement column
		JMP	SETCRS
;
CRET:		MOV	DL,0			;set first column
		JMP	SETCRS
;
LFEED:		CMP	DH,BYTE PTR SCRLIN	;check bottom line
		JNB	SCROLL			;yes - scroll screen
		JMP	INCLIN			;no - increment line
;
BELL:		MOV	AH,0EH			;TTY character out
		INT	010H
		RET		
;
;	VCH - Video Clear Screen / Home Cursor
;		Entry to CBIOS88:  DH = 01CH
;
VCH:		CALL	NORMAL			;reset attribute byte
		CALL	CLEAR			;clear screen/home cursor
		JMP	EXIT
;
;	KEYPAD - Check if keypad key (1234567890.-+) don't translate it.
;
KEYPAD:	EQU	$
	CMP	AL,080H			;check if extended key code
	JNB	KEYIN2R			;do translate on extended code
	CMP	AH,047H			;check scan code below keypad
	JB	KEYIN2R			;yes, go translate it
	JMP	KEYIN5			;if keypad number, don't translate
;
KEYIN2R	JMP	KEYIN2			;return to translate character.
;
;	CKSHIFT - Check keyboard shift status - Do Caps Lock Processing
;
CKSHIFT:	EQU	$
;	Check if Caps Lock is on.
	MOV	BL,SHIFT		;get keyboard shift status
	TEST	BL,040H			;see if Caps Lock is on.
	JNZ	CKS1			;yes - process caps lock codes
	RET				;no - just return
CKS1:	MOV	BH,KEYID		;get which keyboard is installed
	TEST	BL,03H			;is R-L shift key pressed?
	JNZ	CKS3			;yes
;	Caps Lock - Process Special Cases
CKS1F:	CMP	BH,'F'			;is it French keyboard?
	JNZ	CKS1S 			;no
	CMP	AL,'?'			;if ? with caps lock,
	JNZ	CKS1S
	MOV	AL,','			;restore to comma
	RET
CKS1S: 	CMP	BH,'S'			;is it Spanish keyboard?
	JNZ	CKS2			;no
	CMP	AL,0A4H			;if ~n
	JNZ	CKS1S1			;no
	MOV	AL,0A5H			;shift to ~N
	RET
CKS1S1:	CMP	AL,087H			;if ,c
	JNZ	CKS2			;no
	MOV	AL,080H			;shift to ,C
	RET
;	Caps Lock - Fold lowercase to UPPERCASE.
CKS2:	CMP	AL,07AH			;check code above lowercase z
	JA	CKSX			;yes - exit
     	CMP	AL,061H			;check code below lowercase a
	JB	CKSX			;yes - exit
	AND	AL,05FH			;fold a-z to A-Z uppercase
CKSX:	RET		
;	Caps Lock + R-L Shift Pressed - check special cases.
CKS3:	CMP	BH,'F'			;Check if French keyboard.
	JNZ	CKS3S			;no
	CMP	AL,','			;if ,
	JNZ	CKS3S
	MOV	AL,'?'			;restore to ?
	RET
CKS3S:	CMP	BH,'S'			;Check if Spanish keyboard.
	JNZ	CKS4			;no
	CMP	AL,0A5H			;if ~N
	JNZ	CKS3S1			;no			
	MOV	AL,0A4H			;lowercase ~n
	RET
CKS3S1:	CMP	AL,080H			;if ,C
	JNZ	CKS4			;no
	MOV	AL,087H			;lowercase ,c
	RET
;	Caps Lock + L-R Shift Pressed - UPPERCASE to lowercase.
CKS4:	CMP	AL,05AH			;check code above Z
	JA	CKS4X			;yes
	CMP	AL,041H			;check code below A
	JB	CKS4X			;yes
	OR	AL,020H			;force lowercase a-z
CKS4X:	RET	
;	
;
;==========================================================================
;
;	Keyboard Translate Table  (512 bytes to accomodate all combinations) 
;	Installed by the KEYBD.COM utility program.
;
;--------------------------------------------------------------------------
;
		ORG	04000H
;
KEYBD:		EQU	$	
		.BYTE	83H,00H		;IBM null to ASCII null
		.BYTE	D3H,7FH		;IBM delete to ASCII delete
		.BYTE	F2H,10H		;IBM control-prtsc to control-P
		.BYTE	0,0		;logical end of table
;
;
;==========================================================================
;
;	Terminal Emulation Processors (Jump Table)
;
;--------------------------------------------------------------------------
;
		ORG	KEYBD+0200H  
;
TERMTBL:	EQU	$
;
	.BYTE	0,0,0		;end of Keybd table, if all 512 bytes used.
HOM:	JMP	HOME		;Home cursor
BRITE:	JMP	INTBRITE	;Bright intensity
NORMI:	JMP	INTNORM		;Normal intensity
REV:	JMP	VIDREV		;Reverse (toggle) video (background/foreground)
NORMV:	JMP	NORMAL		;Normal Video (background/foregound)
BW:	JMP	BWVIDEO		;Black & White video
BLNK:	JMP	BLINKON		;Blinking Text
NBLNK:	JMP	BLINKOFF	;Normal Text (non-blinking)
UNDLN:	JMP	UNDLNON		;Underline character (monochrome only)
NUNDLN:	JMP	UNDLNOFF	;Underline off (monochrome only)
SAVCRS:	JMP	REMCUR		;Remember (save) cursor position.
RESCRS:	JMP	RESTCUR		;Restore remembered cursor position.
BLOCK:	JMP	BLKCUR		;Block cursor type.
LINE:	JMP	LINCUR		;Line cursor type.
NL:	JMP	NEWLINE		;New line (carriage-return, line-feed).
EEOL:	JMP	ERASEOL		;Erase to End-of-line.
EEOS:	JMP	ERASEOS		;Erase to End-of-screen.
RCUR:	JMP	RIGHTCUR	;Cursor Right one column.
DCUR:	JMP	DOWNCUR		;Cursor Down one line.
UCUR:	JMP	UPCUR		;Cursor Up one line.
SETCUR:	JMP	CURSOR		;Set Cursor Position.
CLHOME:	JMP	CLEAR		;Do Clear/Home.
DMPSTR:	JMP	NULSTR		;Dump string (non-emulated string).
LININS:	JMP	INSERT		;Insert Line at cursor position.
LINDEL:	JMP	DELETE		;Delete Line at cursor position.
;
;==========================================================================
;
;	Terminal Emulation Control Table  (Hewlett-Packard 2621 test table.)
;		Other terminal tables installed by the RTERM utility.
;
;--------------------------------------------------------------------------
;
	ORG	TERMTBL+0100H
;
STRTBL:	EQU	$		;Start of terminal emulation control table. 
;
;				Test table Hewlett-Packard 2621
		.WORD	HPCP1	;cursor position
		.WORD	HPCH1	;clear/home
		.WORD	HPEOS	;erase to end of screen
		.WORD	HPEOL	;erase to end of line
		.WORD	HPLIN	;insert line
		.WORD	HPLDL	;delete line
		.WORD	HPRV1	;reverse video
		.WORD	HPNV1	;normal video
		.WORD	0	;end of strings
HPCP1	.WORD	SETCUR		;set cursor position
	.BYTE	01H,01BH,026H,061H,0FAH,0FBH,0F8H,079H,001H
;                   Esc  &    a   nib1 nib2 line  y   bias
	.BYTE	0FCH,0FDH,0F9H,043H,001H,00H
;               nib1 nib2 col   C   bias
HPCH1	.WORD	CLHOME		;clear/home
	.BYTE	01H,01BH,048H,01BH,04AH,00H
;                   Esc   H   Esc   J
HPEOS	.WORD	EEOS		;erase to end of screen
	.BYTE	01H,01BH,04AH,00H
;                   Esc   J
HPEOL	.WORD	EEOL		;erase to end of line
	.BYTE	01H,01BH,04BH,00H
;		    Esc   K
HPLIN	.WORD	LININS		;insert line
	.BYTE	01H,01BH,04CH,00H
;                   Esc   L
HPLDL	.WORD	LINDEL		;delete line
	.BYTE	01H,01BH,04DH,00H
;                   Esc   M
HPRV1	.WORD	REV		;reverse video
	.BYTE	01H,01BH,026H,064H,041H,00H
;                   Esc   &    d    A
HPNV1	.WORD	NORMV		;normal video
	.BYTE	01H,01BH,026H,064H,040H,00H
;                   Esc   &    d    @
;
;===========================================================================
;
;
	ORG	STRTBL+0300H
;
	.BYTE	'PC RP/M2 Version 2A, (C) 1986 by MicroMethods, Inc.'
;
	END
;
